#region References

using System;
using System.Data;
using System.Text;
using System.Configuration;
using System.IO;
using System.Data.SqlClient;
using System.Reflection;
using System.Xml;

using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary; 

#endregion

namespace gov.va.med.vbecs.DAL.HL7AL
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian Tomlin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>5/10/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary></summary>

	#endregion

	/// <summary>
	/// HL7MessageLog
	/// </summary>
	public class HL7MessageLog
	{
		#region Inner Classes

		/// <summary>
		///
		/// </summary>
		private class DataTableDebugHelper
		{		
			public static string Describe( string dtCodeName, DataTable dataTable )
			{
				if( dtCodeName == null )
					throw( new ArgumentNullException( "dtCodeName" ) );

				if( dataTable == null )
					throw( new ArgumentNullException( "dataTable" ) );

				DataTable dt = dataTable.Copy();			

				StringBuilder _sb = new StringBuilder();

				_sb.AppendFormat( "Describing DataTable {0}{1}", dtCodeName, Environment.NewLine );
				_sb.Append( DescribeTableWithXml( dt ) );
				_sb.Append( Environment.NewLine );			

				return _sb.ToString();
			}

			/// <summary>
			///
			/// </summary>
			private static string DescribeTableWithXml( DataTable dt )
			{
				DataSet _ds = new DataSet( "DataTableDecriptor" );
				_ds.Tables.Add( dt );

				MemoryStream _ms = new MemoryStream( 10000 );
			
				XmlTextWriter _wtr = new XmlTextWriter( _ms, Encoding.ASCII );
				_wtr.Formatting = Formatting.Indented;			
				_ds.WriteXml( _wtr );
				_wtr.Flush();

				_ms.Seek( 0L, SeekOrigin.Begin );			
				StreamReader _rdr = new StreamReader( _ms );			
				string _result = _rdr.ReadToEnd();

				_rdr.Close();
				_wtr.Close();
				_ms.Close();

				return _result;
			}
		}

		#endregion

		#region Constructors

		/// <summary>
		/// Blank constructor for HL7MessageLog
		/// </summary>
		static HL7MessageLog(){}

		#endregion

		#region Public Methods

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8263"> 
		///		<ExpectedInput>Insert Valid HL7 message in MessageLog table.</ExpectedInput>
		///		<ExpectedOutput>New entry created in MessageLog table.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8264"> 
		///		<ExpectedInput>Attempt to insert valid HL7 message without including the 
		///		Message Control ID</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inserts a new entry in the MessageLog table in VBECS_INTERFACE
		/// </summary>
		/// <param name="dtMessageLog">Full HL7 message</param>
		/// <returns>0 indicating successfull insert. Anything else indicates failure.</returns>
		public static int InsertMessageLog( DataTable dtMessageLog )
		{
			if( dtMessageLog == null )
				throw( new ArgumentNullException( "dtMessageLog" ) );

			return new Common.StoredProcedure().TransactionalGetValue(STOREDPROC.HL7InsertMessageLog.StoredProcName, dtMessageLog );
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8265"> 
		///		<ExpectedInput>Update existing entry in the MessageLog table where the MessageStatus is Processing
		///		Incoming Message to Successfully Completed.</ExpectedInput>
		///		<ExpectedOutput>MessageStatus column is correctly updated.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8266"> 
		///		<ExpectedInput>Attempt to update an existing entry in the MessageLog table without
		///		providing the MessageControlId value.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates the Status of an entry in the MessageLog.
		/// CR 2940
		/// </summary>
		/// <param name="status">Status of message</param>
		/// <param name="messageControlID">Unique identifier for message</param>
		/// <param name="errorText">Error text associated with message (may be null)</param>
		/// <param name="transmitCount">Number of times message has been acknowledged (if transmitCount is less than or equal to zero, parameter will be set to DBNull.Value)</param>
		/// <param name="updateUser">Last update user</param>
		/// <param name="functionID">Last update function</param>
		/// <returns>0 indicating successfull update. Anything else indicates failure.</returns>
		public static int UpdateMessageStatus( MessageStatus status, string messageControlID, string errorText, int transmitCount, string updateUser, UpdateFunction functionID )
		{
			if( status.ToString() == null)
				throw( new ArgumentNullException("status"));
			//
			if( messageControlID == null )
				throw( new ArgumentNullException("messageControlID"));
			//
			if( updateUser == null)
				throw( new ArgumentNullException("updateUser"));
			//
			if( functionID.ToString() == null)
				throw( new ArgumentNullException("functionID"));
			//
			SqlParameter[] prms =
				{
					new SqlParameter( STOREDPROC.HL7UpdateMessageStatus.MessageControlID, SqlDbType.VarChar ),
					new SqlParameter( STOREDPROC.HL7UpdateMessageStatus.MessageStatusCode, SqlDbType.Int ),
					new SqlParameter( STOREDPROC.HL7UpdateMessageStatus.ErrorText, SqlDbType.VarChar ),
					new SqlParameter( STOREDPROC.HL7UpdateMessageStatus.TransmitCount, SqlDbType.Int ),
					new SqlParameter( STOREDPROC.HL7UpdateMessageStatus.LastUpdateUser, SqlDbType.VarChar),
					new SqlParameter( STOREDPROC.HL7UpdateMessageStatus.LastUpdateFunctionId, SqlDbType.Int),
			};
			//
			prms[0].Value = messageControlID;
			prms[1].Value = (int)status;
			prms[2].Value = errorText;
			//
			// CR 2940
			if ( transmitCount <= 0 )
			{
				prms[3].Value = DBNull.Value;
			}
			else
			{
				prms[3].Value = transmitCount;
			}
			//
			prms[4].Value = updateUser;
			prms[5].Value = (int)functionID;
			//
			return new Common.StoredProcedure().TransactionalGetValue(STOREDPROC.HL7UpdateMessageStatus.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8267"> 
		///		<ExpectedInput>Insert AckMessageControlId for existing message.</ExpectedInput>
		///		<ExpectedOutput>AckMessageControlId is inserted and value of 0 is returned.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8268"> 
		///		<ExpectedInput>Attempt insert on non-existing MessageControlId.</ExpectedInput>
		///		<ExpectedOutput>Value of 1 is returned to indicate failure.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inserts the Ack Message Control ID for a Message recieved and processed.
		/// </summary>
		/// <param name="msgControl">Unique identifier for original message.</param>
		/// <param name="ackMessageControl">Unique identifier for acknowledgement message.</param>
		/// <param name="status">Status of entry in MessageLog</param>
		/// <param name="updateUser">Last update user</param>
		/// <param name="functionID">Last update function</param>
		/// <returns>0 indicating successfull insert of AckMessageControlId. Anything else indicates failure.</returns>
		public static int InsertAckMessageControlId(string msgControl, string ackMessageControl, MessageStatus status, string updateUser, UpdateFunction functionID )
		{
			DataTable dtAckMsg = new DataTable();
			DataRow drAckMsg = null;
			dtAckMsg.Columns.Add(Common.VbecsTables.MessageLog.AckMessageControlId);
			dtAckMsg.Columns.Add(Common.VbecsTables.MessageLog.MessageControlId);
			dtAckMsg.Columns.Add(Common.VbecsTables.MessageLog.MessageStatusCode);
			dtAckMsg.Columns.Add(Common.VbecsTables.MessageLog.LastUpdateUser);
			dtAckMsg.Columns.Add(Common.VbecsTables.MessageLog.LastUpdateFunctionId);

			drAckMsg = dtAckMsg.NewRow();
			drAckMsg[Common.VbecsTables.MessageLog.AckMessageControlId] = ackMessageControl;
			drAckMsg[Common.VbecsTables.MessageLog.MessageControlId] = msgControl;
			drAckMsg[Common.VbecsTables.MessageLog.MessageStatusCode] = (int)status;
			drAckMsg[Common.VbecsTables.MessageLog.LastUpdateUser] = updateUser;
			drAckMsg[Common.VbecsTables.MessageLog.LastUpdateFunctionId] = (int)functionID;

			dtAckMsg.Rows.Add(drAckMsg);

			return new StoredProcedure().TransactionalGetValue(STOREDPROC.HL7UpdateAckMessageControlID.StoredProcName, dtAckMsg);
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8269"> 
		///		<ExpectedInput>Valid HL7 message.</ExpectedInput>
		///		<ExpectedOutput>Non-null DataTable used to insert values into MessageLog.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8270"> 
		///		<ExpectedInput>Null msg input value</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Method to build DataTable used to create a new entry in the MessageLog based on the message.
		/// </summary>
		/// <param name="msg">HL7 Message string</param>
		/// <param name="status">Message Status</param>
		/// <param name="intParms"></param>
		/// <param name="functionID"></param>
		/// <returns></returns>
		public static DataTable GetMessageDataForMessageLogInsert(string msg, MessageStatus status, HL7Interface intParms, Common.UpdateFunction functionID )
		{
			if ( msg == null )
				throw( new ArgumentNullException( "msg" ) );
			string _errorText = null;

			char[] delimiters = HL7Utility.ParseGetMessageDelimiters( msg );

			// Split on the \r's to reveal the segments
			string[] _hl7Segments = HL7Utility.ParseGetAllMessageSegments( msg );

			// Check if this is an ACK message and if there is error text
			string _msaString = HL7Utility.ParseGetOptionalMessageSegment( msg, SegmentTypeNames.MSA );
			if( _msaString != null )
			{
				_errorText = HL7Utility.ParseGetSegmentData( _msaString, delimiters[0].ToString(), 3 );
			}

			// Split the MSH segment to get control fields
			string[] _msh = _hl7Segments[0].Split(delimiters[0]);

			DataTable dtMessageLog = new DataTable();
		
			DataRow drMessageLog = null;
			dtMessageLog.Columns.Add("MessageControlID");
			dtMessageLog.Columns.Add("InterfaceControlId");
			dtMessageLog.Columns.Add("MessageStatusCode");
			dtMessageLog.Columns.Add("DateTimeOfMessage");
			dtMessageLog.Columns.Add("MessageStream");
			dtMessageLog.Columns.Add("ErrorText");
			dtMessageLog.Columns.Add("MessageType");
			dtMessageLog.Columns.Add("SendingApplication");
			dtMessageLog.Columns.Add("ReceivingApplication");
			dtMessageLog.Columns.Add("LastUpdateUser");
			dtMessageLog.Columns.Add("LastUpdateFunctionId");

			drMessageLog = dtMessageLog.NewRow();
			drMessageLog["MessageControlID"] = _msh[9].ToString();
			drMessageLog["InterfaceControlId"] = intParms.InterfaceControlId;
			drMessageLog["MessageStatusCode"] = (int)status;
            //Defect#: 229650 added to accomodate 24 charactor datetime.
			drMessageLog["DateTimeOfMessage"] = HL7DateFormat.ConvertHL7LongDateTime(_msh[6].ToString());
			drMessageLog["MessageStream"] = msg;
			drMessageLog["ErrorText"] = _errorText != null ? _errorText : string.Empty;
			drMessageLog["MessageType"] = _msh[8].ToString();
			drMessageLog["SendingApplication"] = _msh[2].ToString();
			drMessageLog["ReceivingApplication"] = _msh[4].ToString();
			drMessageLog["LastUpdateUser"] = _msh[2].ToString();
			drMessageLog["LastUpdateFunctionId"] = (int)functionID;
			
			dtMessageLog.Rows.Add(drMessageLog);

			return dtMessageLog;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/9/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8811"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8812"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Purges HL7 messages from the MessageLog table 
		/// based on criteria defined in VBECS Administrator
		/// CR 2410
		/// </summary>
		public static int PurgeMessageLog()
		{
			return new StoredProcedure().SimpleTransactionalGetValue( STOREDPROC.HL7PurgeMessageLog.StoredProcName );
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/16/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8823"> 
		///		<ExpectedInput>messagecontrol id</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8824"> 
		///		<ExpectedInput>null controlid</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2940
		/// </summary>
		public static int GetTransmitCountForMessage(string messageControlId)
		{
			if ( messageControlId == null )
			{
				throw( new ArgumentNullException( "messageControlId" ) );
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.HL7GetTransmitCountForMessage.MessageControlId, System.Data.SqlDbType.VarChar),
			};
			//
			prms[0].Value = messageControlId;
			//
			return new Common.StoredProcedure().GetIntReturnValue( STOREDPROC.HL7GetTransmitCountForMessage.StoredProcName, prms );
		}

		#endregion

        // EventLogAppender: removed couple unused methods...
	}
}
